* Title: 	cex_clean.do
* Version: 	23 May 2018
* Purpose: 	Produce annual file for 2016 Consumer Expenditure Survey


*******************************************************************************
* (0) Start of file
*******************************************************************************

capture log close
log using log/cex_clean, replace
set more off
clear all
macro drop _all


*******************************************************************************
* (1) Load and merge MEMI and FMLI data
*******************************************************************************

use dta/raw/cex/memi161x, clear
append using dta/raw/cex/memi162
append using dta/raw/cex/memi163
append using dta/raw/cex/memi164
append using dta/raw/cex/memi171
keep newid age sex cu_code earner incomey wkstatus arm_forc
destring age sex cu_code earner incomey wkstatus arm_forc, replace

* Number of adults in household
generate adult = (age>=18)
bysort newid: egen H_Size_Adults = total(adult)

* Keep only reference person for Consumer Unit
keep if cu_code == 1
tempfile memi16
save `memi16'

use dta/raw/cex/fmli161x, clear
append using dta/raw/cex/fmli162
append using dta/raw/cex/fmli163
append using dta/raw/cex/fmli164
append using dta/raw/cex/fmli171
keep newid fam_size qint* fincbtxm totxest finatxem *pq *cq finlwt21

merge 1:1 newid using `memi16', assert(matched) nogen


*******************************************************************************
* (2) Adjust population weights for calendar-year data
*******************************************************************************

/*The following steps create three groups for quarters. Since quarter 2, 3, and 
  4 are treated the same, we call them all quarter '3.' Quarter 1 and quarter 5
  have special rules for determining which observations are classified.*/
  
gen quarter = 3
replace quarter = 1 if (qintrvmo == "01" | qintrvmo == "02" | qintrvmo == "03") & qintrvyr == "2016"
replace quarter = 5 if (qintrvmo == "01" | qintrvmo == "02" | qintrvmo == "03") & qintrvyr == "2017"

gen month = real(qintrvmo)

gen cal_popweight = finlwt21*(((month-1)/3)/4) if quarter == 1
replace cal_popweight = finlwt21*(((4-month)/3)/4) if quarter == 5
replace cal_popweight = finlwt21/4 if quarter == 3


*******************************************************************************
* (3) Construct weighted income variables
*******************************************************************************

foreach var in	fincbtxm ///
				totxest ///
				finatxem {
	gen	double 	`var'_weight = `var' *finlwt21/4 				 if quarter == 3
	replace  	`var'_weight = `var' *finlwt21*(((month-1)/3)/4) if quarter == 1
	replace 	`var'_weight = `var' *finlwt21*(((4-month)/3)/4) if quarter == 5
}

rename fincbtxm			income_pretax
rename fincbtxm_weight	income_pretax_weight
rename totxest_weight	total_tax_weight
rename finatxem_weight	income_posttax_weight


*******************************************************************************
* (4) Construct weighted expenditure variables
*******************************************************************************
foreach var in	totexp 	///
				food 	///
				hous 	///
				appar 	///
				trans 	///
				persca	///
				health 	///
				entert 	///
				alcbev 	///
				tobacc {
	gen double	`var'_weight = `var'cq *finlwt21 				if quarter==1
	replace 	`var'_weight = `var'pq *finlwt21 				if quarter==5
	replace 	`var'_weight = ((`var'cq + `var'pq) *finlwt21) 	if quarter==3
					
}
rename totexp_weight	totexp_weight
rename food_weight		food_weight
rename hous_weight		housing_weight
rename appar_weight		apparel_weight
rename trans_weight		transport_weight
rename persca_weight	personalcare_weight
rename health_weight	health_weight
rename entert_weight	entertainment_weight
rename alcbev_weight	alcohol_weight
rename tobacc_weight	tobacco_weight


*******************************************************************************
* (5) End of file
*******************************************************************************

* Drop armed forces
drop if arm_forc==1

keep newid fam_size H_Size_Adults age sex cu_code earner incomey wkstatus arm_forc income_pretax *weight

compress

label data "Create CEX data \ 05-23-2018" 
datasignature set
save dta/cex_clean, replace

log close
exit, clear
